package easik.sketch.util;

import java.io.File;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;

import javax.swing.JOptionPane;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;


import org.w3c.dom.Document;
import org.w3c.dom.Element;

import easik.Easik;
import easik.sketch.Sketch;
import easik.sketch.attribute.EntityAttribute;
import easik.sketch.attribute.UniqueKey;
import easik.sketch.constraint.Constraint;
import easik.sketch.datatype.DataType;
import easik.sketch.datatype.DataTypeController;
import easik.sketch.document.DocumentInfo;
import easik.sketch.edge.SketchEdge;
import easik.sketch.path.SketchPath;
import easik.sketch.util.Export.ExportHandler;
import easik.sketch.vertex.EntityNode;

/**
 * Here is a collection of  static methods which are used to save and load Sketches from
 * XML. 
 * 
 * @author Rob Fletcher 2005
 * @author Kevin Green 2006
 * @author Vera Ranieri 2006
 * @version 2006-07-14 Vera Ranieri
 */
public class SketchFileIO {
	/**
	 * Converts a sketch to an XML file. The empty sketch is provided by the method 
	 * which calls this, it should be empty, or else it will get crowded. Returns the
	 * success of the save.
	 * 
	 * @param outputFile The file we will output to
	 * @param inputSketch The sketch we're reading to
	 * @return True if successful, false otherwise
	 */
	public static boolean sketchToXML(File outputFile, Sketch inputSketch) {
		Document sketchAsXML;
		System.out.println("Trying to save to " + outputFile + " ...");

		try {
			DocumentBuilderFactory docBuilderFactory =
				DocumentBuilderFactory.newInstance();
			DocumentBuilder db = docBuilderFactory.newDocumentBuilder();
			sketchAsXML = db.newDocument();

			Element rootElement = sketchAsXML.createElement("easketch");
			Element header = sketchAsXML.createElement("header");
			Element entities = sketchAsXML.createElement("entities");
			Element edges = sketchAsXML.createElement("edges");
			Element paths = sketchAsXML.createElement("paths");
			Element constraints = sketchAsXML.createElement("constraints");
			Element datatypes = sketchAsXML.createElement("datatypes");
			
			//Add Header info to document
			DocumentInfo d = inputSketch.getDocInfo();
			Element name = sketchAsXML.createElement("title");
			name.appendChild(sketchAsXML.createTextNode(d.getName()));
			header.appendChild(name);
			
			
			ArrayList<String> auts = d.getAuthors();
			for(String aut : auts){
				Element author = sketchAsXML.createElement("author");
				author.appendChild(sketchAsXML.createTextNode(aut));
				header.appendChild(author);
			}
			
			Element desc = sketchAsXML.createElement("description");
			desc.appendChild(sketchAsXML.createTextNode(d.getDesc()));
			header.appendChild(desc);
			
			Element creationDate = sketchAsXML.createElement("creationDate");
			creationDate.appendChild(sketchAsXML.createTextNode(d.getCreationDate()));
			header.appendChild(creationDate);
			
			Element modDate = sketchAsXML.createElement("lastModificationDate");
			modDate.appendChild(sketchAsXML.createTextNode(d.getLastMod()));
			header.appendChild(modDate);

			// Loop through entities, add them to the document
			Iterator iterator = inputSketch.getEntities().keySet().iterator();
			while (iterator.hasNext()) {
				EntityNode currentEntity = (EntityNode) inputSketch.getEntities().get(iterator.next());
				if (currentEntity != null) {
					Element thisEntity = sketchAsXML.createElement("entity");
					thisEntity.setAttribute("name", currentEntity.toString());
					thisEntity.setAttribute(
						"x",
						(int) currentEntity.getX() + "");
					thisEntity.setAttribute(
						"y",
						(int) currentEntity.getY() + "");
					entities.appendChild(thisEntity);
					
					//Loop through attributes, add them to the document
					int numAttributes = currentEntity.getAttributes().size();
					for(int i=0; i<numAttributes;i++){
						Element attributeElmt = sketchAsXML.createElement("attribute");
						EntityAttribute curAttribute = (EntityAttribute) currentEntity.getAttributes().get(i);
						attributeElmt.setAttribute("name", curAttribute.getName());
						attributeElmt.setAttribute("attributeType", curAttribute.getDataType().getTypeName());
						thisEntity.appendChild(attributeElmt);
					}
					
					//Loop through unique keys, add them to the document
					int numUniqueKeys = currentEntity.getUniqueKeys().size();
					for(int i=0; i<numUniqueKeys; i++){
						Element uniqueKeyElmt = sketchAsXML.createElement("uniqueKey");
						UniqueKey curKey = (UniqueKey) currentEntity.getUniqueKeys().get(i);
						uniqueKeyElmt.setAttribute("name", curKey.getKeyName());
						thisEntity.appendChild(uniqueKeyElmt);
						
						int numKeyAttributes = curKey.getAttributes().size();
						for(int j=0; j<numKeyAttributes;j++){
							Element attributeElmt = sketchAsXML.createElement("attref");
							EntityAttribute curAttribute = (EntityAttribute) curKey.getAttributes().get(j);
							attributeElmt.setAttribute("name", curAttribute.getName());
							uniqueKeyElmt.appendChild(attributeElmt);
						}
					}
				}
			}

			iterator = inputSketch.getEdges().keySet().iterator();
			while (iterator.hasNext()) {
				Element thisEdge = sketchAsXML.createElement("edge");
				SketchEdge currentEdge =
					(SketchEdge) inputSketch.getEdges().get(iterator.next());

				thisEdge.setAttribute("id", currentEdge.getName());
				thisEdge.setAttribute(
					"source",
					Easik.getInstance().getFrame().getSketch().getGraphData().getEdgeSource(currentEdge).toString());
				thisEdge.setAttribute(
					"target",
					Easik.getInstance().getFrame().getSketch().getGraphData().getEdgeTarget(currentEdge).toString());
				thisEdge.setAttribute(
					"injective",
					currentEdge.getInjective() + "");
				edges.appendChild(thisEdge);
			}

			//Don't use predefined collection so that paths aren't recorded if they're not 
			//involved in a constraint.
			HashMap<String, SketchPath> allpaths = new HashMap<String, SketchPath>();
			
			// Now add the constraints and populate temp path storage
			for (int i = 0; i < inputSketch.getConstraints().size(); i++) {
				Constraint curConstraint =(Constraint) inputSketch.getConstraints().get(i);
				Element thisConstraint = sketchAsXML.createElement(curConstraint.getType());
				thisConstraint.setAttribute("x", curConstraint.getX() + "");
				thisConstraint.setAttribute("y", curConstraint.getY() + "");
				String visStr;
				if(curConstraint.isVisible())
					visStr = "true";
				else
					visStr = "false";
				thisConstraint.setAttribute("isVisible", visStr);

				ArrayList curConstraintPaths = curConstraint.getPaths();
				for(int j=0; j<curConstraintPaths.size(); j++){
					SketchPath curConstraintPath = (SketchPath)curConstraintPaths.get(j);
					
					//Add path to paths list
					if(!allpaths.containsKey(curConstraintPath.getId()))
						allpaths.put(curConstraintPath.getId(), curConstraintPath);
					
					//Add pathref to constraint
					Element PathRef = sketchAsXML.createElement("pathref");
					PathRef.setAttribute("id", curConstraintPath.getId());
					thisConstraint.appendChild(PathRef);
				}
				//Add constraint to constraints
				constraints.appendChild(thisConstraint);
			}
			
			//Now add paths

			
			Iterator pathsIter = allpaths.keySet().iterator();
			while(pathsIter.hasNext())
			{
				SketchPath curPath = (SketchPath)allpaths.get(pathsIter.next());
				Element thisPath = sketchAsXML.createElement("path");
				thisPath.setAttribute("domain", curPath.getDomain().getName());
				thisPath.setAttribute("id", curPath.getId());
				for(int i=0; i<curPath.getEdges().size(); i++)
				{
					SketchEdge curEdge = (SketchEdge) curPath.getEdges().get(i);
					Element thisEdge = sketchAsXML.createElement("edgeref");
					thisEdge.setAttribute("id", curEdge.getName());
					thisPath.appendChild(thisEdge);
				}
				paths.appendChild(thisPath);
			}
			
			//Set datatype group
			DataTypeController myCont = Easik.getInstance().getFrame().getSketch().getDataTypeController();
			
			datatypes.setAttribute("MySQL", String.valueOf(myCont.is_useMySQL()));
			datatypes.setAttribute("Oracle", String.valueOf(myCont.is_useOracle()));
			datatypes.setAttribute("DB2", String.valueOf(myCont.is_useDB2()));
			datatypes.setAttribute("XML", String.valueOf(myCont.is_useXML()));
			datatypes.setAttribute("UserDefined", String.valueOf(myCont.is_useUserDefined()));
			
			//Add datatypes
			Iterator datatypesIter = Easik.getInstance().getFrame().getSketch().getDataTypeController().getDataTypes().iterator();
			while(datatypesIter.hasNext()){
				Element curType = sketchAsXML.createElement("datatype");
				DataType curDataType = (DataType) datatypesIter.next();
				curType.setAttribute("name", curDataType.getTypeName());
				curType.setAttribute("desc", curDataType.getTypeDesc());
				if(myCont.is_useMySQL()){
					Element MySQLType = sketchAsXML.createElement("MySQL");
					MySQLType.setAttribute("type", curDataType.get_MySQL_type());
					curType.appendChild(MySQLType);
				}
				if(myCont.is_useOracle()){
					Element OracleType = sketchAsXML.createElement("Oracle");
					OracleType.setAttribute("type", curDataType.get_Oracle_type());
					curType.appendChild(OracleType);
				}
				if(myCont.is_useDB2()){
					Element DB2Type = sketchAsXML.createElement("DB2");
					DB2Type.setAttribute("type", curDataType.get_DB2_type());
					curType.appendChild(DB2Type);
				}
				if(myCont.is_useXML()){
					Element XMLType = sketchAsXML.createElement("XML");
					XMLType.setAttribute("type", curDataType.get_XML_type());
					curType.appendChild(XMLType);
				}
				if(myCont.is_useUserDefined()){
					Element UserDefinedType = sketchAsXML.createElement("UserDefined");
					UserDefinedType.setAttribute("type", curDataType.get_user_type());
					curType.appendChild(UserDefinedType);
				}
				datatypes.appendChild(curType);
			}
			
			//Add root elements to document
			sketchAsXML.appendChild(rootElement);
			rootElement.appendChild(header);
			rootElement.appendChild(entities);
			rootElement.appendChild(edges);
			rootElement.appendChild(paths);
			rootElement.appendChild(constraints);
			rootElement.appendChild(datatypes);

		} catch (Exception e) {
			System.out.println("Didn't work...");
			e.printStackTrace();
			sketchAsXML = null;
		}

		outputXMLtoFile(outputFile, sketchAsXML);
		return true;
	}

	/**
	 * Output the document as XML
	 * 
	 * @param outputFile output file
	 * @param xml output XML	  
	 */
	private static void outputXMLtoFile(File outputFile, Document xml) {
		try {
			TransformerFactory tFactory = TransformerFactory.newInstance();
			Transformer transformer = tFactory.newTransformer();
			transformer.setOutputProperty(OutputKeys.INDENT, "yes");
			transformer.transform(
				new DOMSource(xml),
				new StreamResult(outputFile));
		} catch (Exception e) {
			System.out.println("Error exporting data.");
		}
	}

	/**
	 * Load a sketch from XML.
	 * 
	 * @param inputFile the file from which the XML will be read
	 * @param outputSketch The sketch which will be getting the new values
	 * @return Returns the success or failure of the reading.
	 */
	public static boolean graphicalSketchFromXML(File inputFile, Sketch outputSketch) {
		
		SketchHandler sketchHandler = new SketchHandler();
		initializeSketchHandlerFromXML(inputFile, sketchHandler);
		//TODO: fix new header part when ready.
		outputSketch.initialiseFromData(
			sketchHandler.getEntities(),
			sketchHandler.getEdges(),
			sketchHandler.getPaths(),
			sketchHandler.getConstraints(),
			sketchHandler.getDataTypes(),
			sketchHandler.getDocumentInfo(),
			sketchHandler.is_useMySQL(),
			sketchHandler.is_useOracle(),
			sketchHandler.is_useDB2(),
			sketchHandler.is_useXML(),
			sketchHandler.is_useUserDefined()
			);
		return true;
	}
	
	/**
	 * Method to initialize a SketchHandler for a supplied XML file
	 * 
	 * @param inputFile The XML file containing the sketch information
	 * @param sketchHandler An instance of a sketchHandler.
	 * @return true if SketchHandler was initialized, false if an exception occurred.
	 * 
	 * @since 2006-05-17 Vera Ranieri
	 */
	public static boolean initializeSketchHandlerFromXML(File inputFile, SketchHandler sketchHandler){
	
		SAXParser parser;
		SAXParserFactory parseFactory = SAXParserFactory.newInstance();
		try {
			parser = parseFactory.newSAXParser();
			parser.parse(inputFile, sketchHandler);
		} catch (Exception e) {
			System.out.println("Could not open XML file for loading");
			e.printStackTrace();
			return false;
		}
		return true;
	}
	
	/**
	 * Method to initialize an ExportHandler from a supplied XML file
	 * 
	 * @param inputFile The XML file containing the information
	 * @param eHandler An instance of an ExportHandler
	 * @return true if SQLHandler was initialized, false if an exceptionn occurred.
	 * 
	 * @since 2006-05-18 Vera Ranieri
	 */
	public static boolean initializeExportHandlerFromXML(File inputFile, ExportHandler eHandler){
		SAXParser parser;
		SAXParserFactory parseFactory = SAXParserFactory.newInstance();
		try {
			parser = parseFactory.newSAXParser();
			parser.parse(inputFile, eHandler);
		} catch (Exception e) {
			JOptionPane.showMessageDialog(null, "Could not open XML file for SQL generation",
					"Error", JOptionPane.ERROR_MESSAGE);
			e.printStackTrace();
			return false;
		}
		return true;
	}
}
